// LocalStorage keys
const LS = {
  theme: "lotr_theme",
  font: "lotr_font",
  track: "lotr_track",
  username: "lotr_username"
};

// Get elements
const audio = document.getElementById("audio");
const playBtn = document.getElementById("playBtn");
const pauseBtn = document.getElementById("pauseBtn");
const stopBtn = document.getElementById("stopBtn");
const trackSelect = document.getElementById("trackSelect");
const fontSelect = document.getElementById("fontSelect");
const usernameInput = document.getElementById("username");
const themeBtn = document.getElementById("themeBtn");
const themeButtons = document.querySelectorAll(".theme-btn");
const currentTimeEl = document.getElementById("currentTime");
const durationEl = document.getElementById("duration");
const seek = document.getElementById("seek");
const coverImg = document.querySelector("#cover img");
const greeting = document.getElementById("greeting");
const siteTitle = document.getElementById("siteTitle");

// Theme images
const themes = {
  dark: { cover: "images/shire.jpg" },
  light: { cover: "images/rivendell.jpg" },
  green: { cover: "images/elf.jpg" }
};

function save(key, value) {
  localStorage.setItem(key, value);
}

function applyTheme(name) {
  document.documentElement.setAttribute("data-theme", name);
  save(LS.theme, name);
  if (themes[name] && themes[name].cover) {
    coverImg.src = themes[name].cover;
  }
}

function load() {
  // Theme
  let theme = localStorage.getItem(LS.theme);
  if (!theme) theme = "dark";
  applyTheme(theme);

  // Font
  let font = localStorage.getItem(LS.font);
  if (!font) font = "Cinzel, serif";
  let found = false;
  for (let i = 0; i < fontSelect.options.length; i++) {
    if (fontSelect.options[i].value === font) {
      found = true;
      break;
    }
  }
  if (found) {
    fontSelect.value = font;
  } else {
    let opt = document.createElement("option");
    opt.value = font;
    opt.text = font.split(",")[0];
    fontSelect.appendChild(opt);
    fontSelect.value = font;
  }
  document.body.style.fontFamily = font;

  // Track
  let track = localStorage.getItem(LS.track);
  if (!track) track = "audio/shire.mp3";
  let trackFound = false;
  for (let i = 0; i < trackSelect.options.length; i++) {
    if (trackSelect.options[i].value === track) {
      trackFound = true;
      break;
    }
  }
  if (trackFound) {
    trackSelect.value = track;
  } else {
    let opt = document.createElement("option");
    opt.value = track;
    opt.text = "Custom Track";
    trackSelect.appendChild(opt);
    trackSelect.value = track;
  }
  audio.src = track;

  // Username
  let username = localStorage.getItem(LS.username);
  if (!username) username = "";
  usernameInput.value = username;
  updateGreeting(username);
}

function updateGreeting(name) {
  if (name && name.trim().length > 0) {
    greeting.textContent = "Welcome, " + name.trim();
  } else {
    greeting.textContent = "";
  }
}

// Play, pause, stop
playBtn.addEventListener("click", function () {
  audio.play();
});
pauseBtn.addEventListener("click", function () {
  audio.pause();
});
stopBtn.addEventListener("click", function () {
  audio.pause();
  audio.currentTime = 0;
});

// Track change
trackSelect.addEventListener("change", function () {
  audio.src = trackSelect.value;
  audio.load();
  save(LS.track, trackSelect.value);
});

// Seek and progress
audio.addEventListener("loadedmetadata", function () {
  durationEl.textContent = formatTime(audio.duration || 0);
  seek.max = Math.floor(audio.duration || 0);
});
audio.addEventListener("timeupdate", function () {
  currentTimeEl.textContent = formatTime(audio.currentTime || 0);
  seek.value = Math.floor(audio.currentTime || 0);
});
seek.addEventListener("input", function () {
  audio.currentTime = seek.value;
});

function formatTime(sec) {
  sec = Number(sec) || 0;
  let m = Math.floor(sec / 60);
  let s = Math.floor(sec % 60);
  if (s < 10) s = "0" + s;
  return m + ":" + s;
}

// Font select
fontSelect.addEventListener("change", function () {
  let val = fontSelect.value;
  document.body.style.fontFamily = val;
  save(LS.font, val);
  let nameNow = usernameInput.value || localStorage.getItem(LS.username) || "";
  if (nameNow) save(LS.username, nameNow);
  updateGreeting(nameNow);
});

// Username input
usernameInput.addEventListener("input", function () {
  let v = usernameInput.value;
  save(LS.username, v);
  updateGreeting(v);
});

// Theme buttons
for (let i = 0; i < themeButtons.length; i++) {
  themeButtons[i].addEventListener("click", function () {
    applyTheme(themeButtons[i].dataset.theme);
  });
}

// Theme toggle
themeBtn.addEventListener("click", function () {
  let order = ["dark", "light", "green"];
  let current = localStorage.getItem(LS.theme) || "dark";
  let idx = order.indexOf(current);
  let next = order[(idx + 1) % order.length];
  applyTheme(next);
});

// Save on unload
window.addEventListener("beforeunload", function () {
  save(LS.theme, localStorage.getItem(LS.theme) || "dark");
  save(LS.font, document.body.style.fontFamily || fontSelect.value || "Cinzel, serif");
  save(LS.track, audio.src || trackSelect.value || "audio/shire.mp3");
  save(LS.username, usernameInput.value || "");
});

load();